function [pac_stat,pivots_out,randverts,rndcolo_data] = pac_stat(varargin)
%This function calculates the PAC stat, percent above chance 
%statistic, which basically tells you that at the density of two vertices
%lists, is the amount of co-localization above or below random
%co-localization.  
%Synatax:   [pac_stat,pivots_out,randverts,rndcolo_data] = pac_stat('vert',array,'filenames','string',
%                               'path','string','iter',100,'dist',2,'save',1)
%Input:     iter = the number of random iterations.  Default = 10
%           save = save the pivots files.  Default = 1 (on)
%           vert = the vertices around which you want objects to be found,
%               if you do not specify, the function will prompt you to open
%               a set.
%           filenames = the filenames of the vertices that is being
%               compared to.
%           path = the path of those filenames
%           dist = the distance around the vertices one would consider
%               objects.  Default = 0, exactly on top of each other
%           vol = the volume where the vertices reside in [x y z]
%           vox_exc = a list of voxels excluded from the volume.
%           randverts = pregenerated random vertices in cell array format,
%               Default = [] or off, if a cell array is entered, it is
%               assumed to be an array random vertices, if the number 1 is
%               entered, it will trigger a dialogue box that will prompt
%               you to open an .m file containing the randverts cell array.
%Output:    pac_stat = the percent co-localization above random statistic.
%           pivots = the pivots that co-localized in the distance
%           randverts =  the matrix of random vertices
%           colo_data = the number of colocalization and the percent of
%               colocalization for each random vertices set to each input
%               objects list.

[filenames,pathname,iter,vert,dist,sav,vol,vox_exc,randverts] = parse(varargin);

%initiate labeling of matrix (happy now?!)
mxlabel = {'x','y','z','idx'};

%generate random vertices of the same density
parpool
vert_num = size(vert,1);
if isempty(randverts)
    h = waitbar(0,'Generating Random Verticies: 0');
    for j = 1:iter
        waitbar((j/iter),h,['Generating Random Vertices: ',num2str(j)]);   %update progress
        randverts_tmp = [];     %set/reset
        nverts = ceil(vert_num/2);      %initiate
        while size(randverts_tmp,1)<vert_num    %keep going until we satisfy the density number
            cache = round(rand(nverts*2,3).*repmat(vol,nverts*2,1));    %generate the random pivots
            cache = unique(cache,'rows');   %no repeats please
            if ~isempty(vox_exc)     %exclusions
                [cache] = setdiff(cache,vox_exc(:,1:3),'rows'); %remove the excluded voxels
            end
            randverts_tmp = union(randverts_tmp,cache,'rows');
            nverts = vert_num-size(randverts_tmp,1);        %how many verts do we need
        end
        randverts{j} = randverts_tmp(1:vert_num,:); %just the correct number of pivots please
    end
    close(h)
    randgen = 1;
else
    randgen = 0;    %random verts not natively generated
end

[names] = file_sort(filenames);         %parse the filenames
h2 = waitbar(0,'Calculating PAC stat for: ','position',[10 50 275 50]);
for i = 1:size(filenames,2)     %go through each comparision file
    waitbar((i/size(filenames,2)),h2,['Calculating PAC stat for: ',names{i}]);   %update progress
    objects = single(dlmread([pathname,filenames{i}],',',1,0));    %open the file
    objects(isnan(objects(:,1)),:) = [];    %remove NaN just in case
    %now look for coorespondences in vert
    %make the actual comparision
    [a2b,b2a,pivots_tmp,vsidx,mverts] = unique_verts_loco(vert(:,1:3),objects(:,1:3),dist,1);   %turn on store verts
    %parse the data a little
    pivots_out{i} = horzcat(pivots_tmp{1},vsidx{3});  %attach the array index of the matched vertices
    colo_num = size(pivots_out{i},1);     %get the number of colo events
    colo_per = colo_num/vert_num;   %percent matching
    %unique colo instances
    ucolo_num = size(mverts,1);
    ucolo_per = ucolo_num/vert_num;
    %no make the comparision for the random verts
    h3 = waitbar(0,'Comparing to Random Verticies: ');
    for k = 1:iter
        waitbar((k/iter),h3,['Comparing to Random Verticies: ',num2str(k)]);   %update progress
        [a2b,b2a,rndpivots_tmp,vsidx2] = unique_verts_loco(randverts{k},objects(:,1:3),dist);
        %parse the data a little
        rndcolo_data(i).num(k) = size(rndpivots_tmp{1},1);     %get the number of colo events
        rndcolo_data(i).percent(k) = rndcolo_data(i).num(k)/vert_num;   %percent matching
    end
    close(h3)
    %calculate the pac stat
    max_colo = size(objects,1)/vert_num;
    rndcolo_per = mean(rndcolo_data(i).percent);    %get the average random colocalization
    pac = colo_per-rndcolo_per;
    upac = ucolo_per-rndcolo_per;
    pac_stat(i).pacnorm = pac/max_colo;         %pac stat normalized to max possible colocalization for this channel
    pac_stat(i).pac = pac;     %percent above random
    pac_stat(i).num = colo_num;                 %colocalized number
    pac_stat(i).percent = colo_per;             %percent colocalized
    pac_stat(i).averndnum = mean(rndcolo_data(i).num);  %the average number of colocalizations for the random trials
    pac_stat(i).stdrnd =  std(rndcolo_data(i).num);     %tge standard deviation of the colocalization number
    pac_stat(i).rndpercent = rndcolo_per;       %random average
    pac_stat(i).stdrndper =  std(rndcolo_data(i).percent);
    pac_stat(i).vertnum = vert_num;              %vertices number, same
    pac_stat(i).rnditernum = iter;              %number of rand vertices, same now
    pac_stat(i).compobjnum = size(objects,1);   %number of objects in the comparision list
    pac_stat(i).maxcoloper = max_colo;          %maximum colocalization possible
    %unique colo events, in terms of the second channel
    pac_stat(i).upacnorm = upac/max_colo;
    pac_stat(i).upac = upac;
    pac_stat(i).unum = ucolo_num;
    pac_stat(i).upercent = ucolo_per;
    
    if sav  %save the pivot files
        if i==1     %first time through initialize
        dir_path = ['pac_stat'];     %where to save it
        %save out the pivots
        mkdir(pathname,dir_path);      %create output directory
        end
        xls_exp = dataset({pivots_out{1,i},mxlabel{:}});               %to export as a XLS need to make the data a dataset.
        export(xls_exp,'File',[[pathname,dir_path],filesep,[filenames{1,i}(1:end-4),'.',num2str(size(pivots_out{1,i},1)),'.csv']],'delimiter',',')
    end
end
close(h2)
if sav
    %save out the pac_stat
    pac_data = struct2cell(pac_stat);
    pac_data = cell2mat(pac_data);
    pac_data = reshape(pac_data,16,size(pac_data,3),1);  %flatten the array
    row_name = {'pacnorm' 'pac' 'num' 'percent' 'averndnum' 'stdrnd' 'rndpercent' 'stdrndper' 'vertnum' 'rnditernum' 'compobjnum' 'maxcoloper'...
        'upacnorm' 'upac' 'unum' 'upercent'};
    %[names] = file_sort(filenames);         %parse the filenames
    pac_dataset = dataset({pac_data,names{:}},'obsname',row_name);
    sav2csv(pac_dataset,'pac_stat.csv',[pathname,dir_path]);     %save out
    if randgen
        save([pathname,dir_path,filesep,'randverts.mat'],'randverts');
    end
end
delete(gcp('nocreate'))

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [filenames,pathname,iter,vert,dist,sav,vol,vox_exc,randverts] = parse(input)

sav = 1;
filenames = [];
pathname = [];
iter = 10;
dist = 0;
vert = [];
vol = [];
vox_exc = [];
randverts = [];

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'iter'
                    iter = input{1,i+1};
                case 'filenames'
                    filenames = input{1,i+1};
                case 'path'
                    pathname = input{1,i+1};
                case 'vert'
                    vert = input{1,i+1};
                case 'dist'
                    dist = input{1,i+1};
                case 'save'
                    sav = input{1,i+1};
                case 'vol'
                    vol = input{1,i+1};
                case 'vox_exc'
                    vox_exc = input{1,i+1};
                case 'randverts'
                    randverts = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%now make sure we have vertices
if isempty(vert)
    [file1,dir1,filterindex] = uigetfile2({'*.csv','Vertices file (*.csv)';'*.*','All Files';},...
        'Open the vertices file','Multiselect','off');
    vert = single(dlmread([dir1,file1],',',1,0));    %open the file
    vert(isnan(vert(:,1)),:) = [];    %remove NaN just in case
end

%get the names of the comparision vertices files
if isempty(filenames)||isempty(pathname)
    [filenames,pathname,filterindex] = uigetfile2({'*.csv','Vertices to compare (*.csv)';'*.*','All Files';},...
        'Comparision Vert files','Multiselect','on');
    %make sure the stks is a cell array
    if ~iscell(filenames)
        filenames = {filenames};
    end
end
%set get the image volume
if isempty(vol)
    [file1,dir1,filterindex] = uigetfile2({'*.csv','image data file (*.csv)';'*.*','All Files';},...
        'Open image data file','Multiselect','off');
    vol = single(dlmread([dir1,file1],',',1,0));    %open the file
end
%now for the random verts
if ~isempty(randverts)
    if ~iscell(randverts)  %open the file
        [file1,dir1,filterindex] = uigetfile2({'*.mat','Random Vertices (*.mat)';'*.*','All Files';},...
            'Open random vertices','Multiselect','off');
        load([dir1,file1]);
    end
    iter = size(randverts,2);
else    %ask if exclusion is required
    %get the exclusion volume, if desired
    if isempty(vox_exc)
        [file1,dir1,filterindex] = uigetfile2({'*.csv','Excluded voxels (*.csv)';'*.*','All Files';},...
            'Open excluded voxel file','Multiselect','off');
        try
            vox_exc = single(dlmread([dir1,file1],',',1,0));    %open the file
        end
    end
end

%-----------------------------------------------------------------------------------------------------------------------------
function [a2b,b2a,ab,idx,mverts] = unique_verts_loco(a,b,nhood,mvt)

if nargin<3
    nhood = 0;  %default = perfect colo
    mvt = [];    %don't record the matched verts
elseif nargin<4
    mvt = [];    %don't record the matched verts
end

b2a = [];
mverts = [];

%lets make the two data sets the same data type, or else the comparision
%will not be true, especially the case for double and single
if ~isinteger(a) && ~isinteger(b)     %if both are integer they are fine if class is different
    a_type = whos('a');
    b_type = whos('b');
    if ~strcmp(a_type,b_type)
        if isinteger(a)||isinteger(b)   %one number is integer make sure everything is integer
            if isfloat(a)
                a = int16(a);
            elseif isfloat(b)
                b = int16(b);
            end
        else    %both numbers a floats, make sure we don't compare double to single, go down to single.
            if isa(b,'double')
                b = single(b);
            elseif isa(a,'double')
                a = single(a);
            end
        end
    end
end
%find the unique verts in a
a_idx = zeros(size(a,1),1);     %preallocate
ab_idx = zeros(size(a,1),1);
%h = waitbar(0,['Analyzing Array A: ']);    %initialize progress bar.

b_tmp = b(:,1:3);
parfor i = 1:size(a,1)     %remember verts are arranged in vertical arrays
    match = find_neighbors(a(i,1:3),b_tmp,nhood);
    if isempty(match)    %unique
        a_idx(i,1) = i;
    else    %not unique
        ab_idx(i,1) = i;
        if ~isempty(mvt)    %if signaled to store the matche verts
            mverts = vertcat(mverts,match);
        end
    end
    %waitbar(i/size(a,1),h,['Analyzing Array A: ',num2str(i)]); %update progress
end
%close(h)

%now create the unique array
a_idx(a_idx==0) = [];       %remove the zeros
ab_idx(ab_idx==0) = [];
a2b = a(a_idx,:);         %unique a vertices
ab{1,1} = sortrows(a(ab_idx,:));         %not unique, should be the same as b
idx{1,1} = a_idx;   %unique a index
idx{1,3} = ab_idx;  %not unique a index
if ~isempty(mvt)
    mverts = unique(mverts,'rows');
end

% %find the unique verts in b
% b_idx = zeros(size(b,1),1);     %preallocate
% ab_idx = zeros(size(b,1),1);
% %h2 = waitbar(0,['Analyzing Array B: ']);    %initialize progress bar.
% parfor j = 1:size(b,1)     %remember verts are arranged in vertical arrays
%     match = find_neighbors(b(j,1:3),a(:,1:3),nhood);
%     if isempty(match)    %unique
%         b_idx(j,1) = j;
%         else    %not unique
%         ab_idx(j,1) = j;
%     end
%     %waitbar(j/size(b,1),h2,['Analyzing Array B: ',num2str(j)]); %update progress
% end
% %close(h2)
% %now create the unique array
% b_idx(b_idx==0) = [];       %remove the zeros
% ab_idx(ab_idx==0) = [];
% b2a = b(b_idx,:);         %unique a vertices
% ab{1,2} = sortrows(b(ab_idx,:));         %not unique, should be the same as b
% idx{1,2} = b_idx;   %unique b index
% idx{1,4} = ab_idx;  %not unique b index

%-----------------------------------------------------------------------------------------------------------------------------
function [uq_names] = file_sort(filenames)

for i = 1:size(filenames,2)      %step through each filename and pull the wanted word
    filename_tmp = filenames{i};
    strmask = isstrprop(filename_tmp,'punct');  %find the punctuations, we only want the last two
    strmask2 = filename_tmp=='&';       %we are going to exempt the & character
    strmask3 = isstrprop(filename_tmp,'wspace'); %include the white space as well
    strmask = strmask-strmask2+strmask3;     %final mask including space and punct, no &
    [x,y] = find(strmask==1);  %get the positions
    %if size(y,2)<=3
        %figure out the end of the file names, remove all numbers
        a = 1;  %initiate
        b = 0;
        tmp = filename_tmp(1,y(end-a)+1:y(end-b)-1);    %create an array of the words
        while ~isempty(str2num(tmp))   %not empty = numbers or vertices file, push forward one and go
            a = a+1;
            b = b+1;
            if size(y,2)-a==0
                break
            end
            tmp = filename_tmp(1,y(end-a)+1:y(end-b)-1);    %create an array of the words
        end
        %now do the same for the front
        a = 0;
        tmp = filename_tmp(1,1:y(1+a)-1);
        while ~isempty(str2num(tmp))&&size(y,2)~=a+1   %not empty = numbers or vertices file, push forward one and go
            a = a+1;
            tmp = filename_tmp(1,1:y(1+a)-1);    %create an array of the words
        end
        if a==0
            uq_names{i,:} = filename_tmp(1,1:y(end-b)-1);
        else
            uq_names{i,:} = filename_tmp(1,y(a)+1:y(end-b)-1);
        end
%     else    %too complicated take the whole thing
%         tmp = filename_tmp(1,1:y(end)-1);    %create an array of the words
%         %now lets consider the vertices case
%         uq_names{i,:} = tmp;
    %end
end